"""
Created on 2019年6月28日
@author: 刘益宗
screen 服务
"""
import math
from copy import deepcopy

from H_9U.models.result import get_result_model, ResInfo
from H_9U.mao.impl import screen
from H_9U.util.cache import cacher
from H_9U.models.cachekey import CacheKey, CacheDuration
from H_9U.mao.output_mao import two_in_one_card_outputs, read_output_origin_resolution, read_dl_outputs, read_4k_outputs
from H_9U.mao.device_mao import read_two_in_one_card_payload, device_api
from H_9U.models.sysconst import ModelId, Output12GSDIConst


class ScreenApi:

    def __read_tio_outputs(self, device_id):
        """
        读取二合一卡slot和对应的output
        :param device_id: 设备id
        :return: {slotid1: [output1, output2, output3, output4]}
        """
        return two_in_one_card_outputs(device_id)

    def __read_dl_outputs(self, device_id):
        """
        读取DL output slot和对应的output
        :param device_id: 设备id
        :return: {slotid1: [output1, output2, output3, output4]}
        """
        return read_dl_outputs(device_id)

    def __read_4k_outputs(self, device_id):
        """
           读取4k output slot和对应的output
           :param device_id: 设备id
           :return: {slotid1: [output1, output2, output3, output4]}
           """
        return read_4k_outputs(device_id)

    def __read_tio_resolution(self, device_id, output_ids):
        """
        读取二合一卡分辨率， 四个接口一致返回成功，否在返回失败
        :param device_id: 设备Id
        :param output_id:  outputId
        :return: width, height, refreshrate
        """
        data = {}
        for o_id in output_ids:
            rs = read_output_origin_resolution(device_id, o_id)
            if rs['status'] == 0:
                if data:
                    if rs['data']['refreshRate'] != data['refreshRate'] \
                            or rs['data']['width'] != data['width'] or rs['data']['height'] != data['height']:
                        return get_result_model(ResInfo.Output_Timing_Err)
                else:
                    data = rs['data']
            else:
                return rs
        rs = get_result_model()
        rs['data'] = data
        return rs

    def __read_dl_output_resolution(self, device_id, output_id):
        rs = read_output_origin_resolution(device_id, output_id)
        if rs['status'] == 0:
            data = {
                'refreshRate':rs['data']['refreshRate'],
                'width': rs['data']['width'],
                'height': rs['data']['height']
            }
            rs = get_result_model()
            rs['data'] = data
            return rs
        else:
            return rs

    def read_list(self, device_id):
        """
        读取屏幕列表并加入缓存
        :param device_id:  设备id
        :return: 屏幕列表
        """
        key = CacheKey.screen_list(device_id)
        return cacher.ScreenCache.try_get_value(key, CacheDuration.Default, screen.screen_list_read, device_id)

    def read_detail(self, device_id, screen_id):
        """
        读取屏幕详情
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return:
        """
        rs = screen.screen_full_detail_read(device_id, screen_id)
        if rs['status'] == 0:
            interfaces = rs['data']['outputMode']['screenInterfaces']
            # 有二合一卡接口
            tio_slots = [x['slotId'] for x in interfaces
                         if x['modelId'] in ModelId.Two_In_One_Card_List]
            if tio_slots:
                tio_slots = set(tio_slots)
                # new_inters 所有非二合一卡的接口
                new_inters = [x for x in interfaces if x['modelId']
                              not in ModelId.Two_In_One_Card_List]
                for slot_id in tio_slots:
                    # 所有二合一卡的interface
                    tios = [x for x in interfaces if
                            x['slotId'] == slot_id
                            and x['modelId'] in ModelId.Two_In_One_Card_List]
                    tios.sort(key=lambda i: i['interfaceId'])

                    if slot_id != 255:  # 非无效slot，正常二合一卡
                        # 通slot下所有二合一卡接口,interfaceId从小到大排序
                        if len(set([x['x'] for x in tios])) == 1 and len(set([x['cropx'] for x in tios])) == 1:
                            direction = 1
                        else:
                            direction = 0
                        new_inter = None
                        for tio in tios:
                            if new_inter is None:  # 第一个接口
                                new_inter = tio
                            else:
                                # if new_inter['y'] == tio['y']:  # 水平
                                if direction == 0:
                                    new_inter['cropx'] += tio['cropx']
                                    new_inter['width'] += tio['width']
                                else:  # 垂直
                                    new_inter['cropy'] += tio['cropy']
                                    new_inter['height'] += tio['height']

                        new_inters.append(new_inter)
                    else:  # 无效接口
                        # 二合一卡的起始下标
                        index = 0
                        for _ in range(len(tios)//4):  # 包含二合一卡的个数
                            new_inter = None
                            for i in range(4):  # interfaceid，从小到打，四个一组
                                tio = tios[index]
                                if new_inter is None:
                                    new_inter = tio
                                else:
                                    if new_inter['y'] == tio['y']:  # 水平
                                        new_inter['cropx'] += tio['cropx']
                                        new_inter['width'] += tio['width']
                                    else:  # 垂直
                                        new_inter['cropy'] += tio['cropy']
                                        new_inter['height'] += tio['height']
                                index += 1

                            new_inters.append(new_inter)
                rs['data']['outputMode']['screenInterfaces'] = new_inters

            # 有DL output
            interfaces = rs['data']['outputMode']['screenInterfaces']
            dl_interfaces = [x for x in interfaces
                        if x['modelId'] in ModelId.DL_Output_Card_List and x['isDL'] == 1]
            if dl_interfaces:
                # new_inters 所有非DL的接口
                new_inters = [x for x in interfaces if ('isDL' not in x or x['isDL'] != 1)]
                dl_interfaces.sort(key=lambda x: x['dlId'], reverse=True)
                for i in range(len(dl_interfaces)//2):
                    new_inter = dl_interfaces[i*2]
                    new_inter['cropx'] += dl_interfaces[i*2+1]['cropx']
                    new_inter['width'] += dl_interfaces[i*2+1]['width']
                    new_inters.append(new_inter)
                rs['data']['outputMode']['screenInterfaces'] = new_inters
            # 有12 G output
            SDI_12_slots = []
            for x in interfaces:
                if x['modelId'] == ModelId.Output_12_3G_SDI_Card and x['resolution']['height'] >= Output12GSDIConst.LimitHeight \
                        and x['resolution']['width'] >= Output12GSDIConst.LimitWidth \
                        and x['resolution']['refresh'] >= Output12GSDIConst.LimitRefresh:
                    SDI_12_slots.append(x['slotId'])
            if SDI_12_slots:
                SDI_12_slots = set(SDI_12_slots)
                # new_inters 所有非二合一卡的接口
                new_inters = [x for x in interfaces if x['modelId'] not in[ModelId.Output_12_3G_SDI_Card]]
                for slot_id in SDI_12_slots:
                    # 所有二合一卡的interface
                    k4s = [x for x in interfaces if
                            x['slotId'] == slot_id
                            and x['modelId'] in [ModelId.Output_12_3G_SDI_Card]]
                    k4s.sort(key=lambda i: i['interfaceId'])

                    if slot_id != 255:  # 非无效slot，正常二合一卡
                        # 通slot下所有二合一卡接口,interfaceId从小到大排序

                        new_inter = None
                        for k4 in k4s:
                            if new_inter is None:  # 第一个接口
                                new_inter = k4
                            else:
                                # if new_inter['y'] == tio['y']:  # 水平
                                new_inter['cropx'] += k4['cropx']
                                new_inter['width'] += k4['width']

                        new_inters.append(new_inter)
                    else:  # 无效接口
                        # 二合一卡的起始下标
                        index = 0
                        for _ in range(len(k4s)//4):  # 包含二合一卡的个数
                            new_inter = None
                            for i in range(4):  # interfaceid，从小到打，四个一组
                                k4 = k4s[index]
                                if new_inter is None:
                                    new_inter = k4
                                else:
                                    new_inter['cropx'] += k4['cropx']
                                    new_inter['width'] += k4['width']

                                index += 1

                            new_inters.append(new_inter)
                rs['data']['outputMode']['screenInterfaces'] = new_inters
            k4_slots = [x['slotId'] for x in interfaces
                        if x['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]]
            if k4_slots:
                k4_slot = set(k4_slots)
                # new_inters 所有非二合一卡的接口
                new_inters = [x for x in interfaces if
                              x['modelId'] not in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]]
                for slot_id in k4_slot:
                    # 所有二合一卡的interface
                    k4s = [x for x in interfaces if
                           x['slotId'] == slot_id
                           and x['modelId'] in [ModelId.Output_2_HDMI20_Card, ModelId.Pd_Output_2_HDMI20_Card]]
                    k4s.sort(key=lambda i: i['interfaceId'])

                    if slot_id != 255:  # 非无效slot，正常二合一卡
                        # 通slot下所有二合一卡接口,interfaceId从小到大排序

                        new_inter = None
                        for k4 in k4s:
                            if new_inter is None:  # 第一个接口
                                new_inter = k4
                            else:
                                # if new_inter['y'] == tio['y']:  # 水平
                                new_inter['cropx'] += k4['cropx']
                                new_inter['width'] += k4['width']

                        new_inters.append(new_inter)
                    else:  # 无效接口
                        # 二合一卡的起始下标
                        index = 0
                        for _ in range(len(k4s)//4):  # 包含二合一卡的个数
                            new_inter = None
                            for i in range(4):  # interfaceid，从小到打，四个一组
                                k4 = k4s[index]
                                if new_inter is None:
                                    new_inter = k4
                                else:
                                    new_inter['cropx'] += k4['cropx']
                                    new_inter['width'] += k4['width']

                                index += 1

                            new_inters.append(new_inter)
                rs['data']['outputMode']['screenInterfaces'] = new_inters
        return rs

    def write_create(self, device_id, data):
        """
        创建屏幕
        清除全部缓存
        :param device_id: 设备id
        :param data: 屏幕信息
        :return: 结果对象
        """

        # 1.判断当前卡是否为12GSDI；
        #    是：
        #      1.是否达到拼接条件；
        #           是：组装接口，在创建屏幕
        #           否：直接创建屏幕
        #    否：走原先的处理逻辑即可

        interfaces = data['screenInterfaces']
        for x in interfaces:
            if x['outputId'] == 255:
                continue
            slot_id = x['slotId']
            slot_rs = device_api.read_slot(device_id, slot_id)
            if slot_rs['status'] != 0:
                continue
            model_id = slot_rs['data']['modelId']
            if model_id == ModelId.Output_12_3G_SDI_Card:
                if x['resolution']['height'] >= Output12GSDIConst.LimitHeight \
                        and x['resolution']['width'] >= Output12GSDIConst.LimitWidth \
                        and x['resolution']['refresh'] >= Output12GSDIConst.LimitRefresh:
                    rs = self._build_sdl_interface(device_id, data['screenInterfaces'])
                    if rs['status'] != 0:
                        return rs
                    data['screenInterfaces'] = rs['data']
                rs = screen.screen_create_write(device_id, data)
                if rs['status'] == 0:
                    cacher.clear_all()
                    return rs
        rs = self._build_screen_interfaces(device_id, data['screenInterfaces'])
        if rs['status'] == 0:
            data['screenInterfaces'] = rs['data']
            rs = screen.screen_create_write(device_id, data)
            if rs['status'] == 0:
                cacher.clear_all()
        return rs

    def _get_twoinone_card_sorted_output(self, output_id, tio_outputs_dict):
        """
        判断output是否是二合一卡的output
        :param outputId: outputId
        :param tio_outputs_dict: 二合一卡 {slotId:[output1, output2, ...]}
        :return: (slotId, sorted(output_list))
        """
        for key, value in tio_outputs_dict.items():
            # 如果能找到interface的output，说明是二合一卡interface
            if output_id in value:
                value.sort()
                return key, value


    def _build_screen_interfaces(self, device_id, screen_interfaces):
        """
        组合屏幕 interface
        :param device_id: 设备Id
        :param screen_interfaces: 屏幕interface
        :return:
        """
        # 获取所有二合一卡和对应的output
        rs = self.__read_tio_outputs(device_id)
        dl_rs = self.__read_dl_outputs(device_id)
        k4_rs = self.__read_4k_outputs(device_id)
        # 查询设备是否有二合一卡\dl output失败，直接返回
        if rs['status'] != 0 or dl_rs['status'] != 0 or k4_rs['status'] != 0:
            return rs
        # 设备没有二合一卡,没有dl output，直接返回
        if not rs['data'] and not dl_rs['data']['dlList'] and not k4_rs['data']:
            rs['data'] = screen_interfaces
            return rs

        # 二合一卡逻辑
        tio_dict = rs['data']  # {slotId:[output1, output2, ...]}
        new_list = []  # 存储屏幕所有interface
        """
            new list保存要创建屏幕的interface
                非二合一卡output构成的interface，直接加入的new list中
                二合一卡output构成的interface，进行拆分后，加入new list 中
            使用new list 创建屏幕
        """
        interface_id = 0
        for interface in screen_interfaces:  # 循环屏幕所有接口
            # 获取二合一卡的solt和output数据
            slot_info = self._get_twoinone_card_sorted_output(interface['outputId'], tio_dict)
            if slot_info is None:  # 不是二合一卡output，直接加入到新列表中
                interface['interfaceId'] = interface_id
                interface_id += 1
                new_list.append(interface)
            else:  # 是二合一卡output
                slot_id, output_ids = slot_info
                rs = self.__read_tio_resolution(device_id, output_ids)
                # 查询output分辨率失败，直接返回
                if rs['status'] != 0:
                    return rs
                slot_rs = read_two_in_one_card_payload(device_id, slot_id)
                # 查询二合一卡分辨率信息，失败直接返回
                if slot_rs['status'] != 0:
                    return slot_rs

                if slot_rs['data']['direction'] == 0:  # 水平拼接
                    output_key = 'width'
                    interface_key = 'width'
                    crop_key = 'cropx'
                    direction = 'x'
                else:
                    output_key = 'height'
                    interface_key = 'height'
                    crop_key = 'cropy'
                    direction = 'y'

                output_data = rs['data'][output_key]  # 单个output的值
                interface_data = interface[interface_key]  # 屏幕interface的总值
                interface_crop_data = interface[crop_key]  # 相对整个屏幕interface的截取点

                # 拆分二合一卡的interface
                next_val = None
                for x in range(len(output_ids)):
                    new_inter = deepcopy(interface)
                    new_inter['interfaceId'] = interface_id
                    interface_id += 1
                    new_inter['outputId'] = output_ids[x]
                    # 接口剩余的值大于0，表示接口为分割完
                    if interface_data > 0:                        
                        offset = x * output_data  # 位移x个接口的宽度/高度

                        if interface_crop_data - offset > output_data:  # 截取位置在output的右边/上边
                            new_inter[crop_key] = output_data
                            new_inter[interface_key] = 0
                        elif interface_crop_data - offset >= 0:  # 截取位置在output内
                            new_inter[crop_key] = interface_crop_data - offset  # 截取位置相对于output的起始位置
                            new_inter[interface_key] = min(offset + output_data - interface_crop_data, interface_data)   # 截取位置到output终点的宽度
                        else:  # 截取位置在output的左边
                            new_inter[crop_key] = 0
                            new_inter[interface_key] = min(output_data, interface_data)

                        if next_val is None:
                            next_val = new_inter[direction] + new_inter[interface_key]
                        else:
                            new_inter[direction] = next_val
                            next_val += new_inter[interface_key]

                        interface_data -= new_inter[interface_key]
                    else:
                        new_inter[crop_key] = 0
                        new_inter[interface_key] = 0
                        new_inter[direction] = next_val

                    new_list.append(new_inter)

        # DL output 逻辑
        screen_interfaces = new_list
        inter_list = []
        interface_id = 0
        for interface in screen_interfaces:  # 循环屏幕所有接口
            if interface['outputId'] not in dl_rs['data']['dlList']:
                interface['interfaceId'] = interface_id
                interface_id += 1
                inter_list.append(interface)
            else:
                rs = self.__read_dl_output_resolution(device_id, interface['outputId'])
                # 查询output分辨率失败，直接返回
                if rs['status'] != 0:
                    return rs

                output_key = 'width'
                interface_key = 'width'
                crop_key = 'cropx'
                direction = 'x'

                output_data = rs['data'][output_key]  # 单个output的值
                interface_data = interface[interface_key]  # 屏幕interface的总值
                interface_crop_data = interface[crop_key]  # 相对整个屏幕interface的截取点

                # 拆分二合一卡的interface
                output_ids = [interface['outputId'], interface['outputId']-1]
                next_val = None
                for x in range(2):
                    new_inter = deepcopy(interface)
                    new_inter['isDL'] = 1
                    new_inter['dlId'] = output_ids[x]
                    new_inter['interfaceId'] = interface_id
                    interface_id += 1
                    new_inter['outputId'] = output_ids[x]
                    # 接口剩余的值大于0，表示接口为分割完
                    if interface_data > 0:
                        offset = x * output_data  # 位移x个接口的宽度/高度
                        if interface_crop_data - offset > output_data:  # 截取位置在output的右边/上边
                            new_inter[crop_key] = output_data
                            new_inter[interface_key] = 0
                        elif interface_crop_data - offset >= 0:  # 截取位置在output内
                            new_inter[crop_key] = interface_crop_data - offset  # 截取位置相对于output的起始位置
                            new_inter[interface_key] = min(offset + output_data - interface_crop_data,
                                                           interface_data)  # 截取位置到output终点的宽度
                        else:  # 截取位置在output的左边
                            new_inter[crop_key] = 0
                            new_inter[interface_key] = min(output_data, interface_data)

                        if next_val is None:
                            next_val = new_inter[direction] + new_inter[interface_key]
                        else:
                            new_inter[direction] = next_val
                            next_val += new_inter[interface_key]

                        interface_data -= new_inter[interface_key]
                    else:
                        new_inter[crop_key] = 0
                        new_inter[interface_key] = 0
                        new_inter[direction] = next_val

                    inter_list.append(new_inter)

        screen_interfaces = inter_list
        #4k卡逻辑
        k4_dict = k4_rs['data']  # {slotId:[output1, output2, ...]}
        new_list = []  # 存储屏幕所有interface
        """
            new list保存要创建屏幕的interface
                非二合一卡output构成的interface，直接加入的new list中
                二合一卡output构成的interface，进行拆分后，加入new list 中
            使用new list 创建屏幕
        """
        interface_id = 0
        for interface in screen_interfaces:  # 循环屏幕所有接口
            # 获取4k卡的solt和output数据
            slot_info = None
            for key, value in k4_dict.items():
                if interface['outputId'] in value:
                    slot_info = key, value
                    break

            if slot_info is None:  # 不是4k output，直接加入到新列表中
                interface['interfaceId'] = interface_id
                interface_id += 1
                new_list.append(interface)
            else:  # 是二合一卡output
                slot_id, output_ids = slot_info
                rs = self.__read_tio_resolution(device_id, output_ids)
                # 查询output分辨率失败，直接返回
                if rs['status'] != 0:
                    return rs
                slot_rs = read_two_in_one_card_payload(device_id, slot_id)
                # 查询4k卡分辨率信息，失败直接返回
                if slot_rs['status'] != 0:
                    return slot_rs

                if slot_rs['data']['direction'] == 0:  # 水平拼接
                    output_key = 'width'
                    interface_key = 'width'
                    crop_key = 'cropx'
                    direction = 'x'
                else:
                    output_key = 'height'
                    interface_key = 'height'
                    crop_key = 'cropy'
                    direction = 'y'

                output_data = rs['data'][output_key]  # 单个output的值
                interface_data = interface[interface_key]  # 屏幕interface的总值
                interface_crop_data = interface[crop_key]  # 相对整个屏幕interface的截取点

                # 拆分interface
                next_val = None
                for x in range(len(output_ids)):
                    new_inter = deepcopy(interface)
                    new_inter['interfaceId'] = interface_id
                    interface_id += 1
                    new_inter['outputId'] = output_ids[x]
                    # 接口剩余的值大于0，表示接口为分割完
                    if interface_data > 0:
                        offset = x * output_data  # 位移x个接口的宽度/高度

                        if interface_crop_data - offset > output_data:  # 截取位置在output的右边/上边
                            new_inter[crop_key] = output_data
                            new_inter[interface_key] = 0
                        elif interface_crop_data - offset >= 0:  # 截取位置在output内
                            new_inter[crop_key] = interface_crop_data - offset  # 截取位置相对于output的起始位置
                            new_inter[interface_key] = min(offset + output_data - interface_crop_data,
                                                           interface_data)  # 截取位置到output终点的宽度
                        else:  # 截取位置在output的左边
                            new_inter[crop_key] = 0
                            new_inter[interface_key] = min(output_data, interface_data)

                        if next_val is None:
                            next_val = new_inter[direction] + new_inter[interface_key]
                        else:
                            new_inter[direction] = next_val
                            next_val += new_inter[interface_key]

                        interface_data -= new_inter[interface_key]
                    else:
                        new_inter[crop_key] = 0
                        new_inter[interface_key] = 0
                        new_inter[direction] = next_val

                    new_list.append(new_inter)

        screen_interfaces = new_list

        rs = get_result_model()
        rs['data'] = screen_interfaces
        return rs

    def _build_sdl_interface(self, device_id, screen_interfaces):
        rs = self.__read_4k_outputs(device_id)
        if rs['status'] != 0:
            return rs
        if not rs['data']:
            rs['data'] = screen_interfaces
            return rs
        k4_dict = rs['data']  # {slotId:[output1, output2, ...]}
        new_list = []  # 存储屏幕所有interface
        """
            new list保存要创建屏幕的interface
                非二合一卡output构成的interface，直接加入的new list中
                二合一卡output构成的interface，进行拆分后，加入new list 中
            使用new list 创建屏幕
        """
        interface_id = 0
        for interface in screen_interfaces:  # 循环屏幕所有接口
            # 获取4k卡的solt和output数据
            slot_info = None
            for key, value in k4_dict.items():
                if interface['outputId'] in value:
                    slot_info = key, value
                    break

            if slot_info is None:  # 不是4k output，直接加入到新列表中
                interface['interfaceId'] = interface_id
                interface_id += 1
                new_list.append(interface)
            else:  # 是二合一卡output
                slot_id, output_ids = slot_info
                rs = self.__read_tio_resolution(device_id, output_ids)
                # 查询output分辨率失败，直接返回
                if rs['status'] != 0:
                    return rs
                slot_rs = read_two_in_one_card_payload(device_id, slot_id)
                # 查询4k卡分辨率信息，失败直接返回
                if slot_rs['status'] != 0:
                    return slot_rs

                if slot_rs['data']['direction'] == 0:  # 水平拼接
                    output_key = 'width'
                    interface_key = 'width'
                    crop_key = 'cropx'
                    direction = 'x'
                else:
                    output_key = 'height'
                    interface_key = 'height'
                    crop_key = 'cropy'
                    direction = 'y'

                output_data = rs['data'][output_key]  # 单个output的值
                interface_data = interface[interface_key]  # 屏幕interface的总值
                interface_crop_data = interface[crop_key]  # 相对整个屏幕interface的截取点

                # 拆分interface
                next_val = None
                for x in range(len(output_ids)):
                    new_inter = deepcopy(interface)
                    new_inter['interfaceId'] = interface_id
                    interface_id += 1
                    new_inter['outputId'] = output_ids[x]
                    # 接口剩余的值大于0，表示接口为分割完
                    if interface_data > 0:
                        offset = x * output_data  # 位移x个接口的宽度/高度

                        if interface_crop_data - offset > output_data:  # 截取位置在output的右边/上边
                            new_inter[crop_key] = output_data
                            new_inter[interface_key] = 0
                        elif interface_crop_data - offset >= 0:  # 截取位置在output内
                            new_inter[crop_key] = interface_crop_data - offset  # 截取位置相对于output的起始位置
                            new_inter[interface_key] = min(offset + output_data - interface_crop_data,
                                                           interface_data)  # 截取位置到output终点的宽度
                        else:  # 截取位置在output的左边
                            new_inter[crop_key] = 0
                            new_inter[interface_key] = min(output_data, interface_data)

                        if next_val is None:
                            next_val = new_inter[direction] + new_inter[interface_key]
                        else:
                            new_inter[direction] = next_val
                            next_val += new_inter[interface_key]

                        interface_data -= new_inter[interface_key]
                    else:
                        new_inter[crop_key] = 0
                        new_inter[interface_key] = 0
                        new_inter[direction] = next_val

                    new_list.append(new_inter)

        screen_interfaces = new_list

        rs = get_result_model()
        rs['data'] = screen_interfaces
        return rs

    def write_delete(self, device_id, screen_id, data):
        """
        删除屏幕
        清除全部缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        rs = screen.screen_delete_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.clear_all()
        return rs

    def write_general(self, device_id, screen_id, data):
        """
        设置屏幕基本信息
        清除屏幕列表缓存
        清除预监窗口全部缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 基本信息
        :return: 结果对象
        """
        rs = screen.screen_general_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.ScreenCache.delete(CacheKey.screen_list(device_id))
            cacher.MvrWindowCache.clear()
        return rs

    def write_output_mode(self, device_id, screen_id, data):
        """
        设置屏幕拼接模式
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        # 1.判断当前卡是否为12GSDI；
        #    是：
        #      1.是否达到拼接条件；
        #           是：组装接口，在创建屏幕
        #           否：直接创建屏幕
        #    否：走原先的处理逻辑即可

        interfaces = data['screenInterfaces']
        for x in interfaces:
            if x['outputId'] == 255:
                continue
            slot_id = x['slotId']
            slot_rs = device_api.read_slot(device_id, slot_id)
            if slot_rs['status'] != 0:
                continue
            model_id = slot_rs['data']['modelId']
            if model_id == ModelId.Output_12_3G_SDI_Card:
                if x['resolution']['height'] >= Output12GSDIConst.LimitHeight \
                        and x['resolution']['width'] >= Output12GSDIConst.LimitWidth \
                        and x['resolution']['refresh'] >= Output12GSDIConst.LimitRefresh:
                    rs = self._build_screen_interfaces(device_id, data['screenInterfaces'])
                    if rs['status'] != 0:
                        return rs
                    data['screenInterfaces'] = rs['data']
                rs = screen.screen_output_mode_write(device_id, screen_id, data)
                if rs['status'] == 0:
                    cacher.clear_all()
                elif rs['status'] == ResInfo.Over_Run.value:
                    rs['status'] = ResInfo.Screen_Over_Run.value
                return rs
        rs = self._build_screen_interfaces(device_id, data['screenInterfaces'])
        if rs['status'] == 0:
            data['screenInterfaces'] = rs['data']
            rs = screen.screen_output_mode_write(device_id, screen_id, data)
            if rs['status'] == 0:
                cacher.clear_all()
            elif rs['status'] == ResInfo.Over_Run.value:
                rs['status'] = ResInfo.Screen_Over_Run.value
        return rs

    def write_ftb(self, device_id, screen_id, data):
        """
        屏幕ftb
        移除屏幕freeze、ftb缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        rs = screen.screen_ftb_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.ScreenCache.delete(CacheKey.screen_freezed_outputs(device_id))
            cacher.LayerCache.clear()
        return rs

    def write_freeze(self, device_id, screen_id, data):
        """
        屏幕冻结
        移除屏幕freeze、ftb缓存
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        rs = screen.screen_freeze_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.ScreenCache.delete(CacheKey.screen_freezed_outputs(device_id))
            cacher.LayerCache.clear()
        return rs

    def write_bkg(self, device_id, screen_id, data):
        """
        屏幕设置bkg
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        return screen.screen_bkg_write(device_id, screen_id, data)

    def write_osd(self, device_id, screen_id, data):
        """
        设置屏幕osd
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        return screen.screen_osd_write(device_id, screen_id, data)

    def write_gamma(self, device_id, screen_id, data):
        """
        设置屏幕gamma
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        rs = screen.screen_gamma_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.OutputCache.clear()
            cacher.ScreenCache.clear()
        return rs

    def write_image_quality(self, device_id, screen_id, data):
        """
        设置屏幕画质
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        rs = screen.screen_image_quality_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.OutputCache.clear()
            cacher.ScreenCache.clear()
        return rs

    def read_screen_brightness(self, device_id, screen_id):
        """
        读取屏幕亮度
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return: 结果对象
        """
        return screen.screen_brightness_read(device_id, screen_id)

    def write_screen_brightness(self, device_id, screen_id, data):
        """
        设置屏幕亮度
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 亮度信息
        :return: 结果对象
        """
        return screen.screen_brightness_write(device_id, screen_id, data)

    def multi_write_brightness(self, device_id, data):
        """
		设置屏幕亮度
		:param device_id: 设备id
		:param screen_id: 屏幕id
		:param data: 亮度信息
		:return: 结果对象
		"""
        return screen.multi_write_brightness(device_id, data)

    def save_screen_brightness(self, device_id, screen_id, data):
        """
        固化屏幕亮度
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: 亮度信息
        :return: 结果对象
        """
        return screen.screen_brightness_save(device_id, screen_id, data)

    def read_screen_hdr(self, device_id, screen_id):
        """
        读取屏幕亮度
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return: 结果对象
        """
        return screen.screen_hdr_read(device_id, screen_id)

    def write_screen_hdr(self, device_id, screen_id, data):
        """
        读取屏幕hdr
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data: hdr信息
        :return: 结果对象
        """
        return screen.screen_hdr_write(device_id, screen_id, data)

    def write_screen_3d(self, device_id, screen_id, data):
        """
        设置屏幕3D
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param data: 3D数据
        :return: 结果对象
        """
        return screen.screen_3d_write(device_id, screen_id, data)

    def write_genlock_phase_offset(self, device_id, screen_id, data):
        """
        设置屏幕相位偏移
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param data: 偏移数据
        :return: 结果对象
        """
        return screen.screen_phase_offset_write(device_id, screen_id, data)

    def write_screen_clear_layers(self, device_id, screen_id, data):
        """
        清空屏幕图层
        :param device_id: 设备Id
        :param screen_id: 屏幕id
        :param data: 清空图层
        :return: 结果对象
        """
        rs = screen.screen_clear_layers(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
        return rs

    def freezed_output(self, device_id):
        """
        屏幕冻结、ftb数据
        :param device_id:
        :return: ftb、freeze屏幕信息
        """
        data = cacher.ScreenCache.get(CacheKey.screen_freezed_outputs(device_id))
        if data:
            return data
        freeze_outputs = []
        ftb_outputs = []
        poll_outputs = []
        data = get_result_model()
        data['data']['freeze'] = freeze_outputs
        data['data']['ftb'] = ftb_outputs
        data['data']['presetPoll'] = poll_outputs
        rs = self.read_list(device_id)
        if rs['status'] != 0 or not rs['data']:
            cacher.ScreenCache.set(CacheKey.screen_freezed_outputs(device_id), data, CacheDuration.Default)
            data = cacher.ScreenCache.get(CacheKey.screen_freezed_outputs(device_id))
            return data
        for screen in rs['data']['screens']:
            screen_id = screen['screenId']
            d_rs = self.read_detail(device_id, screen_id)
            if d_rs['status'] != 0 or not d_rs['data']:
                continue
            o_list = [x['outputId'] for x in d_rs['data']['outputMode']['screenInterfaces']
                      if 0 <= x['outputId'] < 255]
            if d_rs['data']['Freeze']['enable'] == 1:  # 1 Freeze
                freeze_outputs.extend(o_list)
            if d_rs['data']['Ftb']['enable'] == 0:  # 0 Ftb
                ftb_outputs.extend(o_list)
            if d_rs['data']['presetPoll']['enable'] == 1:  # 场景轮循
                poll_outputs.extend(o_list)
        data['data']['freeze'] = freeze_outputs
        data['data']['ftb'] = ftb_outputs
        data['data']['presetPoll'] = poll_outputs
        cacher.ScreenCache.set(CacheKey.screen_freezed_outputs(device_id), data, CacheDuration.Default)
        data = cacher.ScreenCache.get(CacheKey.screen_freezed_outputs(device_id))
        return data

    def write_notime_play(self, device_id, screen_id, data):
        """
        非实时上屏take
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param data: 图层数据
        :return: 结果对象
        """
        rs = screen.screen_notime_play(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
            cacher.DeviceCache.clear()
        return rs

    def read_nontime_detail(self, device_id, screen_id):
        """
        读取非实时上屏全部数据
        :param device_id:设备Id
        :param screen_id:屏幕Id
        :return:结果对象
        """
        rs = screen.screen_nontime_full_detail_read(device_id, screen_id)
        return rs

    def read_multi_nontime_detail(self, device_id, data):
        """
        读取非实时上屏全部数据
        :param device_id:设备Id
        :param screen_id:屏幕Id
        :return:结果对象
        """
        rs = screen.screen_multi_nontime_full_detail_read(device_id, data)
        return rs

    def read_template_list(self, device_id):
        """
        读取屏幕布局模板列表数据
        :param device_id: 设备Id
        :return: 结果对象
        """
        key = CacheKey.screen_template_list(device_id)
        return cacher.ScreenCache.try_get_value(key, CacheDuration.Default, screen.read_template_list, device_id)


    def template_add(self, device_id, data):
        """
        添加屏幕布局模板列表数据
        :param device_id: 设备Id
        :param data: data
        :return: 结果对象
        """
        rs = screen.template_add( device_id, data)
        if rs['status'] == 0:
            cacher.ScreenCache.delete(CacheKey.screen_template_list(device_id))
        return rs

    def template_delete(self, device_id, data):
        """
        删除屏幕布局模板列表数据
        :param device_id: 设备Id
        :param data: data
        :return: 结果对象
        """
        rs = screen.template_delete(device_id, data)
        if rs['status'] == 0:
            cacher.ScreenCache.delete(CacheKey.screen_template_list(device_id))
        return rs

    def read_bright_adj_list(self, device_id, screen_id):
        """
        读取亮度调节列表
        :param device_id:设备Id
        :param screen_id:屏幕Id
        :return:结果对象
        """
        return screen.read_bright_adj_list(device_id, screen_id)

    def write_bright_adj_list(self, device_id, screen_id, data):
        """
        设置亮度调节列表
        :param device_id:设备Id
        :param screen_id:屏幕Id
        :return:结果对象
        """
        return screen.write_bright_adj_list(device_id, screen_id, data)

    def write_audio_attribute(self, device_id, screen_id, data):
        """
        设置屏幕音频属性
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        return screen.write_audio_attribute(device_id, screen_id, data)

    def volume_write(self, device_id, screen_id, data):
        """
        设置屏幕音频属性
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :return: 结果对象
        """
        rs = screen.volume_write(device_id, screen_id, data)
        if rs['status'] == 0:
            cacher.LayerCache.clear()
            cacher.ScreenCache.clear()
        return rs


    def read_bkg(self, device_id, screen_id):
        """
        屏幕设置bkg
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return: 结果对象
        """
        return screen.screen_bkg_read(device_id, screen_id)

    def read_osd(self, device_id, screen_id):
        """
        设置屏幕osd
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :return: 结果对象
        """
        return screen.screen_osd_read(device_id, screen_id)


    def read_osd_weather(self, device_id, screen_id, osd_id):
        """
        读取屏幕天气OSD
        :param device_id: 设备Id
        :param screen_id: 屏幕Id
        :param osd_id: osdId
        :return:
        """
        return screen.screen_osd_weather_read(device_id, screen_id, osd_id)

    def write_osd_and_handle_limit(self, device_id, screen_id, data):
        """
        设置屏幕osd,对返回结果进行判断，是否超过时间OSD限制
        :param device_id: 设备id
        :param screen_id: 屏幕id
        :param data:
        :return: 结果对象
        """
        rs = screen.screen_osd_write(device_id, screen_id, data)
        # 屏幕时间OSD组件超过限制，嵌入式返回特定错误，需要做特殊处理，向前端返回错误码
        if rs['status'] == ResInfo.Verify_Req_Error.value:
            rs = get_result_model(ResInfo.Screen_Time_OSD_Limit_Err)
        return rs

    def write_dynamic_subpixel(self, device_id, data):
        """
        设置屏幕动态子像素
        :param device_id: 设备id
        :param data:
        :return: 结果对象
        """
        return screen.screen_dynamic_subpixel_write(device_id, data)

    def read_detail_batch(self, device_id, data):
        """
        批量获取屏幕详情
        """
        return screen.batch_write_screen_detail(device_id, data)


screen_api = ScreenApi()
