'''
Created on 2019年8月6日
@author: GongQingBao
数据管道：Socket-TCP客户端.
未包含业务逻辑,只处理了以下内容：
    1.建立链接、断线重链;
    2.数据发送、支持失败后重发;
    3.数据接收;
    4.若callbackFunction不为空则建立链接后对其回调;
'''

import socket
import threading
import traceback

from H_9U.util.hidothreading import HidoTimer
from H_9U.protocol.responseSatus import ResponseStatus
from H_9U.util.log import logger


class Pipe(object):


    def __init__(self, ip=None, port=None, timeout=2, callbackFunction=None, linkeCycle=15, delayInitTime=0.05, synModel=False):
        """构造函数
        :param ip:TCP-Server的IP地址[可为None,为空是默认：127.0.0.1]
        :param port:TCP-Server的Port[可为None,为空是默认：8200]
        :param timeout:Socket套接字操作超时时间,单位：秒(浮点数)
        :param callbackFunction:Socket链接建立成功后仅调用一次的回调函数[可为None]
        :param linkeCycle:断线后的重链周期,单位：秒
        :param delayInitTime:TCP延时初始化时间,单位：秒,默认：0.05秒
        :param synModel:初始化是否使用同步模式[True-是|False-否]
        """
        self.ip = '127.0.0.1' if ip == None else ip
        self.port = 8200 if port == None else port
        self.timeout = timeout
        self.callbackFunction = callbackFunction
        self.relinkCycleTime = linkeCycle

        self.linked = False     # 记录socket链接是否成功[True-成功|False-失败]
        self.linking = False      # 记录是否正在进行socket链接工作[True-是|False-否]
        self.theadLinking = False  # 记录是否启动线程进行socket链接工作[True-是|False-否]
        self.sock = None          # Socket-TCP链接对象/实例
        self.BUFF_SIZE = 10240    # 数据接收缓冲区大小
        self.isCallbacked = False # callbackFunction回调函数是否已经调用[True-已调用、False-未调用]

        # 若为同步模式则立即进行链接操作,否则使用timer延迟链接
        if synModel:
            self.circleLinkServerStarter()
        else:
            HidoTimer(delayInitTime, self.circleLinkServerStarter).start()

    def circleLinkServerStarter(self):
        """
        循环重连启动器，保证一个pipe下只有一个线程在进行重连
        :return
        """
        if not self.theadLinking:
            self.theadLinking = True
            self.circleLinkServer()
#     def checkImmediatelyCon(self):
#         """判断是否进行立即链接.
#         :return True-是、False-否.
#         """
#         if not TcpServerSts.delay:
#             return True;
#         else:
#             if TcpServerSts.active:
#                 return True;
#             else:
#                 if TcpServerSts.clientName == self.__str__():
#                     return True;
#                 else:
#                     return False;


#     def recordTcpServerSts(self, active):
#         """记录TCP-Server的状态.
#         :param bln:链接状态[True-存活中、False-灭亡]
#         """
#         if active == False:
#             if TcpServerSts.delay and TcpServerSts.active:
#                 TcpServerSts.active = False;
#                 TcpServerSts.clientName = self.__str__();
#         else:
#             if TcpServerSts.delay:
#                 TcpServerSts.active = True;


    def circleLinkServer(self):
        """与Server进行链接，若链接失败则进入周期重链、直至成功为止.
        """
        logger.info("***** 进入socket重连")

        if self.linked == False:
            self.linkToServer(self.ip, self.port)

        if self.linked == False:
            HidoTimer(self.relinkCycleTime, self.circleLinkServer).start()
            logger.info("***** 启用下一次socket重连Timer")


    def linkToServer(self, ip, port):
        """与Server建立TCP链接
        :param ip:TCP-Server的IP地址
        :param port:TCP-Server的端口
        """
#         self.threadLock.acquire(); # 同步：如果'上次未完成则需进行等待'
        if self.linking == True:
            logger.info("***** socket链接进行中，取消本次链接动作")
            return
        self.linking = True
        self.linked = False

        try:
            if self.sock != None:
                self.sock.shutdown(socket.SHUT_RDWR)
                self.sock.close()
                self.sock = None
        except Exception:
            msg = traceback.format_exc()
            logger.warning("链接前,先断开链接:出现异常!可谓正常!excetion:%s" % msg)
        sock = None

        try:
#             if not self.checkImmediatelyCon():
#                 logger.warning("暂不与TCP-Server进行链接,OBJ:%s" % self.__str__());
#                 return
            logger.debug('尝试与server建立链接..., IP:%s, Port:%i' % (ip, port))

            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(self.timeout)
            sock.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1) # 开启keepAlive
#             sock.setsockopt(socket.SOL_TCP, socket.TCP_KEEPIDLE, 10) # 连接不活跃 10s 后开始 KeepAlive 检测
#             sock.setsockopt(socket.SOL_TCP, socket.TCP_KEEPINTVL, 6) # 每隔 6s 发送一次keepAlive检查
#             sock.setsockopt(socket.SOL_TCP, socket.TCP_KEEPCNT, 3)   # 允许失败的次数
            
            sock.connect((ip, port))
            self.sock = sock
            self.linked = True

            logger.info('与server建立链接:成功! IP:%s, Port:%i, timeout:%s' % (ip, port, str(self.timeout)))
#             self.recordTcpServerSts(True)

            # socket链接成功2秒后进行回调
            if self.linked == True and self.isCallbacked == False:
                if self.callbackFunction != None:
                    self.isCallbacked = True
                    threading.Thread(target=self.callbackFunction).start()
                else:
                    logger.debug("不进行回调:self.callbackFunction为空!")
            else:
                logger.debug("不进行回调! linked:%s, isCallbacked:%s" % (self.linked, self.isCallbacked))
            # 连接成功把连接线程状态置为初值
            self.theadLinking = False
        except Exception:
            self.linked = False
            msg = traceback.format_exc()
            logger.error('与server建立链接:失败-异常! OBJ:%s, IP:%s, Port:%i, timeout:%s, 异常信息Exception:%s' % (self.__str__(), ip, port, str(self.timeout), msg))

            if sock != None:
#                 self.recordTcpServerSts(False)
                try:
                    sock.shutdown(socket.SHUT_RDWR)
                    sock.close()
                    sock = None
                    logger.error('socket打开失败后仍然强制关闭-成功! IP:%s, Port:%i, timeout:%s' % (ip, port, str(self.timeout)))
                except Exception:
                    msg = traceback.format_exc()
                    logger.error('socket打开失败后仍然强制关闭-异常! IP:%s, Port:%i, timeout:%s, 异常信息Exception:%s' % (ip, port, str(self.timeout), msg))
#         self.threadLock.release() # 释放锁
        self.linking = False


    def reLinkToServer(self):
        """与Server重新建立TCP链接(先关闭、再链接)
        """
        self.linked = False

        try:
            self.sock.shutdown(socket.SHUT_RDWR)
            self.sock.close()
            self.sock = None
            logger.warning("链接前,先关闭链接-成功!")
        except Exception:
            msg = traceback.format_exc()
            logger.warning("链接前,先断开链接-出现异常!可谓正常!excetion:%s" % msg)

        if self.linking == False:
            self.circleLinkServerStarter()


    def setTimeout(self, timeoutValue):
        """设置socket超时时间
        :param timeoutValue:通讯超时时间,单位：秒
        """
        if self.sock != None and self.linked:
            try:
                self.sock.settimeout(timeoutValue)
            except Exception:
                msg = traceback.format_exc()
                logger.error("sock.settimeout时出现异常:%s" % msg)
        else:
            logger.error("设置socket超时时间-失败:self.sock为空!")


    def sendData(self, byteA, reSendNum=1):
        """向TCP-Server发送数据
        :param byteA:发送给Server的字节数组
        :param reSendNum:重发次数[0-不重发]
        :return:结果码{数字型},定义：[0-成功、100-超时、101-未建立链接、101-异常]
        """
        code = 0

        if self.linked == True:
            # optType = bytesToIntLH1(byteA[10:11], 0) # 获得操作类型[0-读,1-写]
            try:
                self.sock.sendall(byteA) # 发送数据.使用sendall(.)而不是send(.)
            except socket.timeout:
                code = ResponseStatus.MIDDLEWARE_COMMUNICATION_Timeout
                msg = traceback.format_exc()
                logger.error('发送失败:socket处理超时,timeout:%s,异常信息:%s' % (str(self.sock.gettimeout()), msg))
                self.reLinkToServer()
            except Exception:
                code = ResponseStatus.MIDDLEWARE_COMMUNICATION_FALURE
                msg = traceback.format_exc()
                logger.error('发送失败:socket出现异常,重链次数:%i,异常信息:%s' % (reSendNum, msg))

                # 如果重发次数大于0则需要进行重发否则执行周期重链操作
                if reSendNum > 0:
                    self.linkToServer(self.ip, self.port)
                    return self.sendData(byteA, reSendNum-1)
                else:
                    self.reLinkToServer()
        else:
            code = ResponseStatus.MIDDLEWARE_COMMUNICATION_FALURE
            logger.error("未与Server建立链接,故终止发送消息!")
            if self.linking == False:
                self.circleLinkServerStarter()

        return code


    def receiveData(self):
        """从TCP-Server接收数据
        :return:元组(code,byteA)
            code:结果码{数字型},定义：[0-成功、100-超时、101-未建立链接、39-数据错误/长度为0、101-异常]
            byteA:接收到的数据{字节数组}
        """
        code = 0
        byteA = bytes(0)

        if self.linked == True:
            try:
                byteA = self.sock.recv(self.BUFF_SIZE) # 阻塞接收数据

                if(len(byteA) == 0):
                    logger.error("socket接收数据长度为0!正常情况下是不会出现的,故需关闭socket重链!")
                    code = ResponseStatus.MIDDLEWARE_DATA_FALURE
                    self.reLinkToServer() # 重链
            except socket.timeout:
                msg = traceback.format_exc()
                logger.error('接收失败:socket处理超时,timeout:%s,异常信息:%s' % (str(self.sock.gettimeout()), msg))
                code = ResponseStatus.MIDDLEWARE_COMMUNICATION_Timeout
                self.reLinkToServer() # 重链
            except Exception:
                code = ResponseStatus.MIDDLEWARE_COMMUNICATION_FALURE
                msg = traceback.format_exc()
                logger.error('接收失败:socket出现异常.异常信息:%s' % msg)
                self.reLinkToServer() # 重链
        else:
            code = ResponseStatus.MIDDLEWARE_COMMUNICATION_FALURE
            logger.error("未与Server建立链接,故终止发送消息!")
            if self.linking == False:
                self.circleLinkServerStarter() # 周期重链

        return code, byteA


